//	Altirra - Atari 800/800XL emulator
//	Copyright (C) 2024 Avery Lee
//
//	This program is free software; you can redistribute it and/or modify
//	it under the terms of the GNU General Public License as published by
//	the Free Software Foundation; either version 2 of the License, or
//	(at your option) any later version.
//
//	This program is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//	GNU General Public License for more details.
//
//	You should have received a copy of the GNU General Public License
//	along with this program; if not, write to the Free Software
//	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

#include <stdafx.h>
#include "printerfont.h"

struct ATPrinterFontView {
	uint8 *mpDst;
	uint8 mWidth;
	uint8 mHeight;
	uint8 mCharFirst;

	template<typename T>
	constexpr explicit ATPrinterFontView(T& font)
		: mpDst(font.mColumns)
		, mWidth(font.kWidth)
		, mHeight(font.kHeight)
		, mCharFirst(font.kCharFirst)
	{

	}
};

struct ATPrinterPropFontView {
	uint16 *mpDst;
	uint8 *mpAdv;
	uint8 mWidth;
	uint8 mHeight;
	uint8 mCharFirst;

	template<typename T>
	constexpr explicit ATPrinterPropFontView(T& font)
		: mpDst(font.mColumns)
		, mpAdv(font.mAdvanceWidths)
		, mWidth(font.kWidth)
		, mHeight(font.kHeight)
		, mCharFirst(font.kCharFirst)
	{

	}
};

static constexpr void ATWritePrinterFont(
	const ATPrinterFontView& view,
	uint8 ch,
	auto... lines
)
{
	if (sizeof...(lines) != view.mHeight)
		throw;

	if (ch < view.mCharFirst)
		throw;

	const char *linePtrs[] {
		lines...
	};

	uint8 *dst = &view.mpDst[view.mWidth * (ch - view.mCharFirst)];

	for(;;) {
		for(uint32 column = 0; column < view.mWidth; ++column) {
			uint8 v = 0;

			for(size_t row = 0; row < sizeof...(lines); ++row) {
				char c = *linePtrs[row]++;

				if (c == '#')
					v |= 1 << row;
				else if (c != '.')
					throw;
			}

			*dst++ = v;
		}

		char termCh = *linePtrs[0]++;

		for(size_t row = 1; row < sizeof...(lines); ++row) {
			char c = *linePtrs[row]++;

			if (c != termCh)
				throw;
		}

		if (!termCh)
			break;
	}
}

static constexpr void ATWritePrinterPropFont(
	const ATPrinterPropFontView& view,
	uint8 ch,
	auto... lines
)
{
	if (sizeof...(lines) != view.mHeight)
		throw;

	if (ch < view.mCharFirst)
		throw;

	const char *linePtrs[] {
		lines...
	};

	uint16 *dst = &view.mpDst[view.mWidth * (ch - view.mCharFirst)];
	uint8 *adv = &view.mpAdv[ch - view.mCharFirst];

	for(;;) {
		bool truncated = false;

		*adv = view.mWidth;

		for(uint32 column = 0; column < view.mWidth; ++column) {
			uint32 v = 0;

			if (!truncated) {
				for(size_t row = 0; row < sizeof...(lines); ++row) {
					char c = *linePtrs[row]++;

					if (c == '#')
						v |= 1 << row;
					else if (c != '.') {
						if (row != 0) {
							linePtrs[-ch];		// broken character
							throw;
						}

						--linePtrs[row];
						*adv = column;
						truncated = true;
						break;
					}
				}
			}

			*dst++ = (uint16)v;
		}

		++adv;

		char termCh = *linePtrs[0]++;

		for(size_t row = 1; row < sizeof...(lines); ++row) {
			char c = *linePtrs[row]++;

			if (c != termCh)
				throw;
		}

		if (!termCh)
			break;

		++ch;
	}
}

consteval ATPrinterFont820::ATPrinterFont820()
	: ATPrinterFont{}
{
	ATPrinterFontView fontView(*this);

	ATWritePrinterFont(
		fontView, 0x20,
		".....|..#..|.#.#.|.#.#.|..#..|##...|.#...|..#..|..#..|..#..|..#..|.....|.....|.....|.....|.....",
		".....|..#..|.#.#.|.#.#.|.####|##..#|#.#..|..#..|.#...|...#.|#.#.#|..#..|.....|.....|.....|....#",
		".....|..#..|.#.#.|#####|#.#..|...#.|#.#..|..#..|#....|....#|.#.#.|..#..|.....|.....|.....|...#.",
		".....|..#..|.....|.#.#.|.###.|..#..|.#...|.....|#....|....#|..#..|#####|.....|#####|.....|..#..",
		".....|..#..|.....|#####|..#.#|.#...|#.#.#|.....|#....|....#|.#.#.|..#..|..#..|.....|.....|.#...",
		".....|.....|.....|.#.#.|####.|#..##|#..#.|.....|.#...|...#.|#.#.#|..#..|..#..|.....|.....|#....",
		".....|..#..|.....|.#.#.|..#..|...##|.##.#|.....|..#..|..#..|..#..|.....|.#...|.....|..#..|....."
	);

	ATWritePrinterFont(
		fontView, 0x30,
		".###.|..#..|.###.|#####|...#.|#####|..###|#####|.###.|.###.|.....|.....|...#.|.....|.#...|.###.",
		"#...#|.##..|#...#|....#|..##.|#....|.#...|....#|#...#|#...#|.....|.....|..#..|.....|..#..|#...#",
		"#..##|..#..|....#|...#.|.#.#.|####.|#....|...#.|#...#|#...#|..#..|..#..|.#...|#####|...#.|...#.",
		"#.#.#|..#..|..##.|..##.|#..#.|....#|####.|..#..|.###.|.####|.....|.....|#....|.....|....#|..#..",
		"##..#|..#..|.#...|....#|#####|....#|#...#|.#...|#...#|....#|..#..|..#..|.#...|#####|...#.|..#..",
		"#...#|..#..|#....|#...#|...#.|#...#|#...#|.#...|#...#|...#.|.....|..#..|..#..|.....|..#..|.....",
		".###.|.###.|#####|.###.|...#.|.###.|.###.|.#...|.###.|###..|.....|.#...|...#.|.....|.#...|..#.."
	);

	ATWritePrinterFont(
		fontView, 0x40,
		".###.|..#..|####.|.###.|####.|#####|#####|.####|#...#|.###.|....#|#...#|#....|#...#|#...#|.###.",
		"#...#|.#.#.|#...#|#...#|#...#|#....|#....|#....|#...#|..#..|....#|#..#.|#....|##.##|#...#|#...#",
		"#.#.#|#...#|#...#|#....|#...#|#....|#....|#....|#...#|..#..|....#|#.#..|#....|#.#.#|##..#|#...#",
		"#.###|#...#|####.|#....|#...#|###..|####.|#....|#####|..#..|....#|##...|#....|#...#|#.#.#|#...#",
		"#.##.|#####|#...#|#....|#...#|#....|#....|#..##|#...#|..#..|....#|#.#..|#....|#...#|#..##|#...#",
		"#....|#...#|#...#|#...#|#...#|#....|#....|#...#|#...#|..#..|#...#|#..#.|#....|#...#|#...#|#...#",
		".####|#...#|####.|.###.|####.|#####|#....|.####|#...#|.###.|.###.|#...#|#####|#...#|#...#|.###."
	);

	ATWritePrinterFont(
		fontView, 0x50,
		"####.|.###.|####.|.###.|#####|#...#|#...#|#...#|#...#|#...#|#####|#####|.....|#####|.....|.....",
		"#...#|#...#|#...#|#...#|..#..|#...#|#...#|#...#|#...#|#...#|....#|##...|#....|...##|.....|.....",
		"#...#|#...#|#...#|#....|..#..|#...#|#...#|#...#|.#.#.|.#.#.|...#.|##...|.#...|...##|..#..|.....",
		"####.|#...#|####.|.###.|..#..|#...#|#...#|#.#.#|..#..|..#..|..#..|##...|..#..|...##|.#.#.|.....",
		"#....|#.#.#|#.#..|....#|..#..|#...#|#...#|#.#.#|.#.#.|..#..|.#...|##...|...#.|...##|#...#|.....",
		"#....|#..#.|#..#.|#...#|..#..|#...#|.#.#.|##.##|#...#|..#..|#....|##...|....#|...##|.....|.....",
		"#....|.##.#|#...#|.###.|..#..|.###.|..#..|#...#|#...#|..#..|#####|#####|.....|#####|.....|#####"
	);

	ATWritePrinterFont(
		fontView, 0x60,
		".#...|.....|#....|.....|....#|.....|..##.|.....|#....|..#..|...#.|#....|.##..|.....|.....|.....",
		"..#..|.....|#....|.....|....#|.....|.#...|.....|#....|.....|.....|#....|..#..|.....|.....|.....",
		"...#.|###..|####.|.####|.####|.###.|.#...|.###.|#....|..#..|...#.|#..#.|..#..|.#.#.|#.##.|.###.",
		".....|...#.|#...#|#....|#...#|#...#|####.|#...#|#.##.|..#..|...#.|#.#..|..#..|#.#.#|##..#|#...#",
		".....|.###.|#...#|#....|#...#|#####|.#...|#####|##..#|..#..|...#.|##...|..#..|#.#.#|#...#|#...#",
		".....|#..#.|#...#|#....|#...#|#....|.#...|....#|#...#|..#..|#..#.|#.#..|..#..|#.#.#|#...#|#...#",
		".....|.####|####.|.####|.####|.####|.#...|####.|#...#|.###.|.##..|#..#.|..#..|#.#.#|#...#|.###."
	);

	ATWritePrinterFont(
		fontView, 0x70,
		".....|.....|.....|.....|.#...|.....|.....|.....|.....|.....|.....|...##|..#..|##...|.....",
		".....|.....|.....|.....|.#...|.....|.....|.....|.....|.....|.....|..#..|..#..|..#..|.....",
		"####.|.####|#.##.|.####|####.|#...#|#...#|#...#|#...#|#...#|#####|..#..|..#..|..#..|.#...",
		"#...#|#...#|##..#|#....|.#...|#...#|#...#|#.#.#|.#.#.|#..##|...#.|.#...|..#..|...#.|#.#.#",
		"####.|.####|#....|#####|.#...|#...#|#...#|#.#.#|..#..|.##.#|..#..|..#..|..#..|..#..|...#.",
		"#....|....#|#....|....#|.#..#|#..##|.#.#.|#.#.#|.#.#.|....#|.#...|..#..|..#..|..#..|.....",
		"#....|....#|#....|####.|..##.|.##.#|..#..|.#.#.|#...#|####.|#####|...##|..#..|##...|....."
	);
}

extern constexpr ATPrinterFont820 g_ATPrinterFont820 = ATPrinterFont820();

consteval ATPrinterFont820S::ATPrinterFont820S()
	: ATPrinterFont{}
{
	ATPrinterFontView fontView(*this);

	ATWritePrinterFont(
		fontView, 0x2F,
		".......",
		".......",
		".......",
		".......",
		".......",
		".......",
		"......."
	);

	ATWritePrinterFont(
		fontView, 0x30,
		".#####.|.......|###..#.|.......|.......|.#.####|.##.##.|.......|.##.##.|....##.|.......|.......|...#...|.......|.......|.......",
		"##....#|.......|#..#..#|.#...#.|..#....|#..#..#|#..#..#|......#|#..#..#|#..#..#|.......|.......|..#.#..|..#.#..|.......|.....#.",
		"#.#...#|#....#.|#..#..#|#.....#|..##...|#..#..#|#..#..#|......#|#..#..#|#..#..#|.......|#......|.#...#.|..#.#..|.......|......#",
		"#..#..#|#######|#..#..#|#..#..#|..#.#..|#..#..#|#..#..#|###...#|#..#..#|#..#..#|##.##..|.##.##.|#.....#|..#.#..|#.....#|#.#...#",
		"#...#.#|#......|#..#..#|#..#..#|..#..#.|#..#..#|#..#..#|...#..#|#..#..#|#..#..#|##.##..|.##.##.|.......|..#.#..|.#...#.|...#..#",
		"#....##|.......|#..#..#|#..#..#|#######|.##...#|#..#..#|....#.#|#..#..#|#..#..#|.......|.......|.......|..#.#..|..#.#..|....##.",
		".#####.|.......|#...##.|.##.##.|..#....|.......|.##....|.....##|.##.##.|.#####.|.......|.......|.......|.......|...#...|......."
	);

	ATWritePrinterFont(
		fontView, 0x40,
		"..####.|####...|#.....#|..###..|#.....#|#######|#######|..###..|#######|.......|.#.....|#######|#######|#######|#######|.#####.",
		".#....#|..#.#..|#######|.#...#.|#######|#..#..#|...#..#|.#...#.|...#...|#.....#|#......|..#....|#......|.....#.|.....#.|#.....#",
		"#..##.#|..#..#.|#..#..#|#.....#|#.....#|#..#..#|...#..#|#.....#|...#...|#.....#|#.....#|...#...|#......|....#..|....#..|#.....#",
		"#.#..##|..#...#|#..#..#|#.....#|#.....#|#..#..#|...#..#|#..#..#|...#...|#######|#.....#|..#.#..|#......|...#...|...#...|#.....#",
		"#.###.#|..#..#.|#..#..#|#.....#|#.....#|#..#..#|...#..#|#..#..#|...#...|#.....#|.######|.#...#.|#......|....#..|..#....|#.....#",
		"..#...#|..#.#..|#..#..#|#.....#|#.....#|#.....#|......#|#..#..#|...#...|#.....#|......#|#.....#|#......|.....#.|.#.....|#.....#",
		"...###.|####...|.##.##.|.#...#.|.#####.|.......|.......|.###.#.|#######|.......|......#|.......|.......|#######|#######|.#####."
	);

	ATWritePrinterFont(
		fontView, 0x50,
		".......|..###..|#######|.#..##.|......#|.######|...####|#######|#.....#|.....##|#.....#|.......|......#|.......|...#...|...#...",
		"#######|.#...#.|...#..#|#..#..#|......#|#......|..#....|.#.....|.#...#.|....#..|##....#|#######|.....#.|.......|....#..|..###..",
		"...#..#|#.....#|...#..#|#..#..#|......#|#......|.#.....|..#....|..#.#..|...#...|#.#...#|#######|....#..|#.....#|.....#.|.#.#.#.",
		"...#..#|#.....#|...#..#|#..#..#|#######|#......|#......|...#...|...#...|###....|#..#..#|#.....#|...#...|#.....#|#######|#..#..#",
		"...#..#|#.#...#|..##..#|#..#..#|......#|#......|.#.....|..#....|..#.#..|...#...|#...#.#|#.....#|..#....|#######|.....#.|...#...",
		"....##.|.#...#.|.#.#..#|#..#..#|......#|#......|..#....|.#.....|.#...#.|....#..|#....##|.......|.#.....|#######|....#..|...#...",
		".......|#.###..|#...##.|.##..#.|......#|.######|...####|#######|#.....#|.....##|#.....#|.......|#......|.......|...#...|...#..."
	);
}

extern constexpr ATPrinterFont820S g_ATPrinterFont820S = ATPrinterFont820S();

consteval ATPrinterFont1025::ATPrinterFont1025()
	: ATPrinterFont{}
{
	ATPrinterFontView fontView(*this);

	ATWritePrinterFont(
		fontView, 0x00,
		"......#..|..#......|..#.#...#|......#..|.........|....#....|..#......|..#......|..#.#.#..|.#.....#.|.#.....#.|..#...#..|#.......#|......#..|......#..|.#.....#.",
		"....#....|....#....|#....#.#.|....#....|.........|.#.....#.|....#....|....#....|#......#.|.........|.........|.........|.#.#.#.#.|....#....|....#....|.........",
		"..#.#.#..|.#...#.#.|#.......#|#.#.#.#.#|..#.#.#..|..#.#.#..|..#.#.#..|......#..|.#.......|.........|.#.....#.|..#.#.#..|#.......#|.#.#...#.|..#.#.#..|..#.#.#..",
		".......#.|.#.....#.|#.#.....#|#........|#........|#.......#|#.......#|....#....|..#......|....#....|.#.....#.|.......#.|#.......#|.#.....#.|#.......#|#.......#",
		".#.#.#.#.|.#.....#.|#...#...#|#.#.#.#..|#........|#.......#|#.......#|....#....|#.#.#....|....#....|.#.....#.|.#.#.#.#.|#.......#|.#.....#.|#.......#|#.......#",
		"#......#.|.#.....#.|#.....#.#|#........|..#.#.#..|#.......#|#.......#|....#....|..#.....#|....#....|.#.....#.|#......#.|#.......#|.#.....#.|#.......#|#.......#",
		".#.#.#..#|...#.#.#.|#.......#|#.#.#.#.#|.#.#.....|..#.#.#..|..#.#.#..|....#....|#.#.#.#..|....#....|..#.#.#..|.#.#.#..#|.#.#.#.#.|...#.#.#.|..#.#.#..|..#.#.#.."
	);

	ATWritePrinterFont(
		fontView, 0x10,
		".#.....#.|....#....|....#....|....#....|......#..|..#......|..#.#...#|....#....|...#.#...|..#......|...#.#...|.........|.........|....#....|.........|.........",
		".........|.#.....#.|#.......#|.#.....#.|....#....|....#....|#....#.#.|.#.....#.|...#.#...|....#....|..#.#.#..|.........|....#....|....#....|....#....|....#....",
		"#.......#|..#.#.#..|.#.....#.|.........|..#.#.#..|..#.#.#..|.........|..#.#.#..|..#.#.#..|..#.#.#..|.#.#.#.#.|.........|..#.#.#..|....#....|..#......|......#..",
		"#.......#|.......#.|.#.....#.|....#....|#.......#|#.......#|#..#.#...|#.......#|.......#.|.......#.|#.......#|.........|#...#...#|#...#...#|#.#.#.#.#|#.#.#.#.#",
		"#.......#|.#.#.#.#.|.#.....#.|....#....|#.#.#.#.#|#.#.#.#.#|.#.....#.|#.#.#.#.#|.#.#.#.#.|.#.#.#.#.|#.#.#.#.#|.........|....#....|..#.#.#..|..#......|......#..",
		"#.......#|#......#.|.#.....#.|....#....|#........|#........|.#.....#.|#........|#......#.|#......#.|#.......#|.........|....#....|....#....|....#....|....#....",
		".#.#.#.#.|.#.#.#..#|...#.#.#.|....#....|..#.#.#..|..#.#.#..|.#.....#.|..#.#.#..|.#.#.#..#|.#.#.#..#|#.......#|.........|....#....|.........|.........|........."
	);

	ATWritePrinterFont(
		fontView, 0x20,
		".........|....#....|..#...#..|..#...#..|....#....|.......#.|...#.#...|.#.#.....|......#..|..#......|.........|.........|.........|.........|.........|.......#.",
		".........|....#....|..#...#..|..#...#..|.#.#.#.#.|#.#...#..|..#...#..|...#.....|....#....|....#....|....#....|....#....|.........|.........|.........|......#..",
		".........|....#....|..#...#..|#.#.#.#.#|#...#....|#.#..#...|...#.#...|..#......|...#.....|.....#...|#...#...#|....#....|.........|.........|.........|.....#...",
		".........|....#....|.........|..#...#..|.#.#.#.#.|....#....|..#.#....|.........|...#.....|.....#...|..#.#.#..|#.#.#.#.#|.........|#.#.#.#.#|.........|....#....",
		".........|.........|.........|#.#.#.#.#|....#...#|...#..#.#|.#...#..#|.........|...#.....|.....#...|#...#...#|....#....|.#.#.....|.........|.........|...#.....",
		".........|.........|.........|..#...#..|.#.#.#.#.|..#...#.#|.#....#..|.........|....#....|....#....|....#....|....#....|...#.....|.........|.#.#.....|..#......",
		".........|....#....|.........|..#...#..|....#....|.#.......|..#.#..#.|.........|......#..|..#......|.........|.........|..#......|.........|.#.#.....|.#......."
	);

	ATWritePrinterFont(
		fontView, 0x30,
		".#.#.#.#.|....#....|..#.#.#..|#.#.#.#.#|....#....|#.#.#.#.#|.#.#.#.#.|#.#.#.#.#|.#.#.#.#.|.#.#.#.#.|.........|.........|.......#.|.........|.#.......|..#.#.#..",
		"#.....#.#|..#.#....|#.......#|......#..|...#.....|#........|#........|.......#.|#.......#|#.......#|...#.#...|...#.#...|......#..|.........|..#......|.#.....#.",
		"#....#..#|....#....|........#|....#....|..#...#..|#.#.#.#..|#........|......#..|#.......#|#.......#|...#.#...|...#.#...|.....#...|#.#.#.#.#|...#.....|.......#.",
		"#...#...#|....#....|......#..|......#..|.#....#..|........#|#.#.#.#..|.....#...|..#.#.#..|..#.#.#.#|.........|.........|....#....|.........|....#....|......#..",
		"#..#....#|....#....|....#....|........#|#.....#..|........#|#.......#|....#....|#.......#|........#|...#.#...|...#.#...|.....#...|#.#.#.#.#|...#.....|....#....",
		"#.#.....#|....#....|..#......|#.......#|#.#.#.#.#|#.......#|#.......#|...#.....|#.......#|........#|...#.#...|.....#...|......#..|.........|..#......|.........",
		".#.#.#.#.|..#.#.#..|#.#.#.#.#|..#.#.#..|......#..|.#.#.#.#.|.#.#.#.#.|...#.....|.#.#.#.#.|.#.#.#.#.|.........|....#....|.......#.|.........|.#.......|....#...."
	);

	ATWritePrinterFont(
		fontView, 0x40,
		".#.#.#.#.|..#.#.#..|#.#.#.#..|.#.#.#.#.|#.#.#.#..|#.#.#.#.#|#.#.#.#.#|.#.#.#.#.|#.......#|..#.#.#..|....#.#.#|#.......#|#........|#.......#|#.......#|.#.#.#.#.",
		"#.......#|.#.....#.|.#......#|#.......#|.#.....#.|#........|#........|#.......#|#.......#|....#....|......#..|#.....#..|#........|#.#...#.#|#.#.....#|#.......#",
		"........#|#.......#|.#......#|#........|.#......#|#........|#........|#........|#.......#|....#....|......#..|#...#....|#........|#..#.#..#|#..#....#|#.......#",
		".#.#.#..#|#.......#|.#.#.#.#.|#........|.#......#|#.#.#.#..|#.#.#.#..|#.....#.#|#.#.#.#.#|....#....|......#..|#.#..#...|#........|#...#...#|#...#...#|#.......#",
		"#....#..#|#.#.#.#.#|.#......#|#........|.#......#|#........|#........|#.......#|#.......#|....#....|......#..|#.....#..|#........|#.......#|#....#..#|#.......#",
		"#....#..#|#.......#|.#......#|#.......#|.#.....#.|#........|#........|#.......#|#.......#|....#....|#.....#..|#......#.|#........|#.......#|#.....#.#|#.......#",
		".#.#.#.#.|#.......#|#.#.#.#..|.#.#.#.#.|#.#.#.#..|#.#.#.#.#|#........|.#.#.#.#.|#.......#|..#.#.#..|.#.#.#...|#.......#|#.#.#.#.#|#.......#|#.......#|.#.#.#.#."
	);

	ATWritePrinterFont(
		fontView, 0x50,
		".#.#.#.#.|.#.#.#.#.|.#.#.#.#.|.#.#.#.#.|#.#.#.#.#|#.......#|#.......#|#.......#|.#.....#.|#.......#|.#.#.#.#.|..#.#.#..|.#.......|..#.#.#..|.........|.........",
		".#......#|#.......#|.#......#|#.......#|....#....|#.......#|#.......#|#.......#|..#...#..|.#.....#.|......#..|..#......|..#......|......#..|....#....|.........",
		".#......#|#.......#|.#......#|#........|....#....|#.......#|#.......#|#.......#|...#.#...|..#...#..|.....#...|..#......|...#.....|......#..|...#.#...|.........",
		".#.#.#.#.|#.......#|.#.#.#.#.|.#.#.#.#.|....#....|#.......#|.#.....#.|#...#...#|....#....|...#.#...|....#....|..#......|....#....|......#..|..#...#..|.........",
		".#.......|#...#...#|.#....#..|........#|....#....|#.......#|..#...#..|#..#.#..#|...#.#...|....#....|...#.....|..#......|.....#...|......#..|.#.....#.|.........",
		".#.......|#.....#..|.#.....#.|#.......#|....#....|#.......#|...#.#...|#.#...#.#|..#...#..|....#....|..#......|..#......|......#..|......#..|.........|.........",
		".#.......|.#.#.#..#|.#......#|.#.#.#.#.|....#....|.#.#.#.#.|....#....|.#.....#.|.#.....#.|....#....|.#.#.#.#.|..#.#.#..|.......#.|..#.#.#..|.........|#.#.#.#.#"
	);

	ATWritePrinterFont(
		fontView, 0x60,
		"....#....|.........|.........|.........|.........|.........|.........|.........|.........|.........|.........|.........|.........|.........|.........|.........",
		".........|.........|.#.......|.........|.......#.|.........|.....#.#.|.........|.#.......|....#....|.....#...|.#.......|....#....|.........|.........|.........",
		"....#....|..#.#.#..|.#.......|...#.#.#.|.......#.|..#.#.#..|....#....|.#.#.#.#.|.#.......|.........|.........|.#.....#.|....#....|.#.....#.|#..#.#...|..#.#.#..",
		"....#....|.......#.|.#.#.#.#.|.#.......|.#.#.#.#.|#.......#|.#.#.#.#.|#......#.|.#.#.#.#.|....#....|.....#...|.#...#...|....#....|#..#.#..#|.#.....#.|#.......#",
		"....#....|.#.#.#.#.|.#......#|.#.......|#......#.|#.#.#.#.#|....#....|.#.#.#.#.|.#......#|....#....|.....#...|.#.#.#...|....#....|#...#...#|.#.....#.|#.......#",
		"....#....|#......#.|.#......#|.#.......|#......#.|#........|....#....|.......#.|.#......#|....#....|.#...#...|.#....#..|....#....|#...#...#|.#.....#.|#.......#",
		"....#....|.#.#.#..#|.#.#.#.#.|...#.#.#.|.#.#.#.#.|..#.#.#..|....#....|..#.#.#..|.#......#|....#....|..#.#....|.#.....#.|....#....|#...#...#|.#.....#.|..#.#.#.."
	);

	ATWritePrinterFont(
		fontView, 0x70,
		".........|.........|.........|.........|.........|.........|.........|.........|.........|.........|.........|.#.....#.|....#....|#.#.#.#..|......#..|..#......",
		".........|.........|.........|.........|...#.....|.........|.........|.........|.........|.........|.........|..#.#.#..|....#....|#.#......|....#.#..|..#.#....",
		".#.#.#.#.|.#.#.#.#.|..#..#.#.|.#.#.#.#.|#.#.#.#..|.#.....#.|#.......#|#.......#|.#.....#.|.#.....#.|.#.#.#.#.|.#.#.#.#.|....#....|#...#....|..#.#.#..|..#.#.#..",
		".#......#|#......#.|...#.....|#........|...#.....|.#.....#.|.#.....#.|#...#...#|..#...#..|..#...#..|......#..|#.......#|....#....|#....#...|#.#.#.#..|..#.#.#.#",
		".#......#|#......#.|...#.....|.#.#.#.#.|...#.....|.#.....#.|..#...#..|#..#.#..#|...#.#...|...#.#...|....#....|#.#.#.#.#|....#....|......#..|..#.#.#..|..#.#.#..",
		".#.#.#.#.|.#.#.#.#.|...#.....|........#|...#...#.|.#.....#.|...#.#...|#.#...#.#|..#...#..|....#....|..#......|#.......#|....#....|......#..|....#.#..|..#.#....",
		".#.......|.......#.|...#.....|.#.#.#.#.|....#.#..|..#.#.#..|....#....|.#.....#.|.#.....#.|.#.#.....|.#.#.#.#.|#.......#|....#....|.........|......#..|..#......"
	);
}

extern constexpr ATPrinterFont1025 g_ATPrinterFont1025 = ATPrinterFont1025();

consteval ATPrinterFont1029::ATPrinterFont1029()
	: ATPrinterFont{}
{
	ATPrinterFontView fontView(*this);

	ATWritePrinterFont(
		fontView, 0x00,
		"...#. .#... .#..# ...#. ..... ..#.. .#... .#... ..##. .#.#. .#.#. .#.#. .#.#. ...#. ...#. .#.#.",
		"..#.. ..#.. #.##. ..#.. .#### .#.#. ..#.. ..#.. .#..# ..... ..... ..... ..... ..#.. ..#.. .....",
		".#... ...#. ..... ##### #.... #...# ...#. ...#. .#... ..... ..... ..... .###. .#... .#... .....",
		".###. #..#. ##..# #.... #.... .###. .###. ..#.. ####. ..#.. #..#. .###. #...# #..#. .###. .###.",
		"#..#. #..#. #.#.# ##### .#### #...# #...# ..#.. .#... ..#.. #..#. #..#. #...# #..#. #...# #...#",
		"#..#. #..#. #..## #.... ..#.. #...# #...# ..#.. .#... ..#.. #..#. #..#. #...# #..#. #...# #...#",
		".##.# .##.# #...# ##### .##.. .###. .###. ..#.. ####. ..#.. .##.# .##.# .###. .##.# .###. .###."
	);

	ATWritePrinterFont(
		fontView, 0x10,
		".#.#. ..#.. ..#.. ..#.. ...#. .#... .#..# ..#.. ..#.. .#... ..#.. ..... ..#.. ..#.. ..... .....",
		"..... .#.#. .#.#. .#.#. ..#.. ..#.. #.##. .#.#. ..... ..#.. ..... ..... .###. ..#.. ..#.. ..#..",
		"#...# #...# #...# #...# .###. .###. ..... .###. ..... ...#. .###. ..... #.#.# ..#.. .#... ...#.",
		"#...# .###. #..#. ..#.. #...# #...# #.##. #...# .###. .###. #...# ..... ..#.. ..#.. ##### #####",
		"#...# #..#. #..#. ..#.. ####. ####. .#..# ####. #..#. #..#. ##### ..... ..#.. #.#.# .#... ...#.",
		"#...# #..#. #..#. ..#.. #.... #.... .#..# #.... #..#. #..#. #...# ..... ..#.. .###. ..#.. ..#..",
		".###. .##.# .##.# ..#.. .###. .###. .#..# .###. .##.# .##.# #...# ..... ..#.. ..#.. ..... ....."
	);

	ATWritePrinterFont(
		fontView, 0x20,
		"..... ..#.. .#.#. .#.#. ..#.. ..... .#... ..#.. ...#. .#... ..#.. ..... ..... ..... ..... .....",
		"..... ..#.. .#.#. .#.#. .#### ##..# #.#.. ..#.. ..#.. ..#.. #.#.# ..#.. ..... ..... ..... ....#",
		"..... ..#.. .#.#. ##### #.#.. ##.#. #.#.. .#... .#... ...#. .###. ..#.. ..... ..... ..... ...#.",
		"..... ..#.. ..... .#.#. .###. ..#.. .#... ..... .#... ...#. ..#.. ##### ..... ##### ..... ..#..",
		"..... ..... ..... ##### ..#.# .#.## #.#.# ..... .#... ...#. .###. ..#.. ..#.. ..... ..... .#...",
		"..... ..... ..... .#.#. ####. #..## #..#. ..... ..#.. ..#.. #.#.# ..#.. ..#.. ..... .##.. #....",
		"..... ..#.. ..... .#.#. ..#.. ..... .##.# ..... ...#. .#... ..#.. ..... .#... ..... .##.. ....."
	);

	ATWritePrinterFont(
		fontView, 0x30,
		".###. ..#.. .###. ##### ...#. ##### .#### ##### .###. .###. ..... ..... ....# ..... #.... .###.",
		"#...# .##.. #...# ...#. ..##. #.... #.... ....# #...# #...# ..... ..... ...#. ..... .#... #...#",
		"#..## ..#.. ....# ..#.. .#.#. ####. #.... ...#. #...# #...# ..#.. ..#.. ..#.. ##### ..#.. ....#",
		"#.#.# ..#.. ...#. ...#. #..#. ....# ####. ..#.. .###. .#### ..... ..... .#... ..... ...#. ...#.",
		"##..# ..#.. ..#.. ....# ##### ....# #...# .#... #...# ....# ..#.. ..#.. ..#.. ##### ..#.. ..#..",
		"#...# ..#.. .#... #...# ...#. #...# #...# .#... #...# ....# ..... ..#.. ...#. ..... .#... .....",
		".###. .###. ##### .###. ...#. .###. .###. .#... .###. ####. ..... .#... ....# ..... #.... ..#.."
	);

	ATWritePrinterFont(
		fontView, 0x40,
		".###. .###. ####. .###. ###.. ##### ##### .#### #...# .###. ..### #...# #.... #...# #...# .###.",
		"#...# #...# #...# #...# #..#. #.... #.... #.... #...# ..#.. ...#. #..#. #.... ##.## #...# #...#",
		"#..## #...# #...# #.... #...# #.... #.... #.... #...# ..#.. ...#. #.#.. #.... #.#.# ##..# #...#",
		"#.#.# ##### ####. #.... #...# ####. ####. #..## ##### ..#.. ...#. ##... #.... #.#.# #.#.# #...#",
		"#.##. #...# #...# #.... #...# #.... #.... #...# #...# ..#.. ...#. #.#.. #.... #...# #..## #...#",
		"#.... #...# #...# #...# #..#. #.... #.... #...# #...# ..#.. #..#. #..#. #.... #...# #...# #...#",
		".#### #...# ####. .###. ###.. ##### #.... .#### #...# .###. .##.. #...# ##### #...# #...# .###."
	);

	ATWritePrinterFont(
		fontView, 0x50,
		"####. .###. ####. .###. ##### #...# #...# #...# #...# #...# ##### .#### ..... ####. ..#.. .....",
		"#...# #...# #...# #...# ..#.. #...# #...# #...# #...# #...# ....# .#... #.... ...#. .#.#. .....",
		"#...# #...# #...# #.... ..#.. #...# #...# #...# .#.#. .#.#. ...#. .#... .#... ...#. #...# .....",
		"####. #...# ####. .###. ..#.. #...# .#.#. #.#.# ..#.. ..#.. ..#.. .#... ..#.. ...#. ..... .....",
		"#.... #.#.# #.#.. ....# ..#.. #...# .#.#. #.#.# .#.#. ..#.. .#... .#... ...#. ...#. ..... .....",
		"#.... #..#. #..#. #...# ..#.. #...# ..#.. ##.## #...# ..#.. #.... .#... ....# ...#. ..... .....",
		"#.... .##.# #...# .###. ..#.. .###. ..#.. #...# #...# ..#.. ##### .#### ..... ####. ..... #####"
	);

	ATWritePrinterFont(
		fontView, 0x60,
		".#... ..... #.... ..... ....# ..... ...## ..... #.... ..#.. ...#. #.... .##.. ..... ..... .....",
		"..#.. ..... #.... ..... ....# ..... ..#.. ..... #.... ..... ..... #.... ..#.. ..... ..... .....",
		"...#. .###. ####. .###. .#### .###. ..#.. .###. #.##. .##.. ...#. #..#. ..#.. ##.#. #.##. .###.",
		"..... ...#. #...# #...# #...# #...# .###. #..#. ##..# ..#.. ...#. #.#.. ..#.. #.#.# .#..# #...#",
		"..... .###. #...# #.... #...# ####. ..#.. .###. #...# ..#.. ...#. ##... ..#.. #.#.# .#..# #...#",
		"..... #..#. #...# #.... #...# #.... ..#.. ...#. #...# ..#.. .#.#. #.#.. ..#.. #.#.# .#..# #...#",
		"..... .##.# ####. .#### .#### .###. ..#.. .###. #...# .###. ..#.. #..#. ..##. #...# .#..# .###."
	);

	ATWritePrinterFont(
		fontView, 0x70,
		"..... ..... ..... ..... .#... ..... ..... ..... ..... ..... ..... ...#. ..#.. .#... ..... #####",
		"..... ..... ..... ..... .#... ..... ..... ..... ..... ..... ..... ..#.. ..#.. ..#.. .#..# #####",
		".###. .###. .#.## .###. ####. #..#. #...# #...# #...# #..#. ##### ..#.. ..#.. ..#.. #.##. #####",
		".#..# #..#. .##.. #.... .#... #..#. #...# #...# .#.#. #..#. ...#. .#... ..#.. ...#. ..... #####",
		".#..# #..#. .#... .###. .#... #..#. .#.#. #.#.# ..#.. .###. ..#.. ..#.. ..#.. ..#.. ..... #####",
		".###. .###. .#... ....# .#..# #..#. .#.#. #.#.# .#.#. ...#. .#... ..#.. ..#.. ..#.. ..... #####",
		".#... ...#. .#... ####. ..##. .##.# ..#.. .#.#. #...# ###.. ##### ...#. ..#.. .#... ..... #####"
	);
}

extern constexpr ATPrinterFont1029 g_ATPrinterFont1029 = ATPrinterFont1029();

////////////////////////////////////////////////////////////////////////////////

consteval ATPrinterFont825Prop::ATPrinterFont825Prop()
	: ATPrinterPropFont{}
{
	ATPrinterPropFontView fontView(*this);

	// Note: All characters have 2 columns of space after them. The widths below
	// +2 add up to the widths given in the 820 Printer Operator's Manual, table
	// 4.

	ATWritePrinterPropFont(
		fontView, 0x20,
		".....|..#..|.#...#..|....#...#....|....#.....|.#.#..........|..#.#.#.....|...#.|...#.|.#...|..........|..........|.....|..........|.....|........#.",
		".....|..#..|.#...#..|....#...#....|.#.#.#.#..|#...#...#.....|.#.....#....|..#..|..#..|..#..|..#...#...|....#.....|.....|..........|.....|......#...",
		".....|..#..|.#...#..|.#.#.#.#.#.#.|#...#.....|.#.#...#......|..#.........|.#...|.#...|...#.|...#.#....|....#.....|.....|..........|.....|.....#....",
		".....|..#..|........|....#...#....|.#.#.#.#..|......#.......|...#......#.|.....|.#...|...#.|#.#...#.#.|#.#.#.#.#.|.....|#.#.#.#.#.|.....|....#.....",
		".....|.....|........|.#.#.#.#.#.#.|....#...#.|.....#...#.#..|.#...#...#..|.....|.#...|...#.|...#.#....|....#.....|.....|..........|.....|...#......",
		".....|.....|........|....#...#....|.#.#.#.#..|....#...#...#.|#......#....|.....|..#..|..#..|..#...#...|....#.....|..#..|..........|..#..|..#.......",
		".....|..#..|........|....#...#....|....#.....|.........#.#..|.#.#.#...#..|.....|...#.|.#...|..........|..........|.#.#.|..........|.#.#.|#.........",
		".....|.....|........|.............|..........|..............|............|.....|.....|.....|..........|..........|..#..|..........|..#..|..........",
		".....|.....|........|.............|..........|..............|............|.....|.....|.....|..........|..........|.#...|..........|.....|.........."
	);

	ATWritePrinterPropFont(
		fontView, 0x30,
		"...#.#....|....#.....|..#.#.#...|..#.#.#...|....#.#...|#.#.#.#.#.|....#.....|#.#.#.#.#.|..#.#.#...|..#.#.#...|..#..|..#..|......#.#.|..........|#.#.......|..#.#.#...",
		".#.....#..|..#.#.....|#.......#.|#.......#.|...#..#...|#.........|...#......|.......#..|#.......#.|#.......#.|.#.#.|.#.#.|....#.....|..........|....#.....|#.......#.",
		"#.......#.|....#.....|.......#..|........#.|..#...#...|#.#.#.#...|..#.......|......#...|#.......#.|#.......#.|..#..|..#..|..#.......|#.#.#.#.#.|......#...|........#.",
		"#.......#.|....#.....|......#...|....#.#...|.#....#...|........#.|.#..#.#...|.....#....|..#.#.#...|..#.#..#..|.....|.....|#.........|..........|........#.|......#...",
		"#.......#.|....#.....|....#.....|........#.|#.#.#.#.#.|........#.|#.......#.|....#.....|#.......#.|......#...|..#..|..#..|..#.......|#.#.#.#.#.|......#...|....#.....",
		".#.....#..|....#.....|..#.......|#.......#.|......#...|#.......#.|#.......#.|...#......|#.......#.|....#.....|.#.#.|.#.#.|....#.....|..........|....#.....|..........",
		"...#.#....|..#.#.#...|#.#.#.#.#.|..#.#.#...|......#...|..#.#.#...|..#.#.#...|..#.......|..#.#.#...|..#.......|..#..|..#..|......#.#.|..........|#.#.......|....#.....",
		"..........|..........|..........|..........|..........|..........|..........|..........|..........|..........|.....|.#...|..........|..........|..........|..........",
		"..........|..........|..........|..........|..........|..........|..........|..........|..........|..........|.....|.....|..........|..........|..........|.........."
	);

	ATWritePrinterPropFont(
		fontView, 0x40,
		"..#.#.#.#...|......#.......|#.#.#.#.#.#..|..#.#.#.#...|#.#.#.#.#.#...|#.#.#.#.#.#.|#.#.#.#.#.#.|..#.#.#.#.#...|#.#.#...#.#.#.|#.#.#.#.|..#.#.#.#.#.|#.#.....#.#.#.|#.#.#.......|#.#.........#.#.|#.#.....#.#.#.|..#.#.#.#.#...",
		".#.......#..|.....#.#......|..#........#.|.#.......#..|..#........#..|..#.......#.|..#.......#.|.#.........#..|..#.......#...|...#....|......#.....|..#.....#.....|..#.........|..#.#.....#.#...|..#.#.....#...|.#.........#..",
		"#.........#.|....#...#.....|..#........#.|#...........|..#.........#.|..#.........|..#.........|#.............|..#.......#...|...#....|......#.....|..#....#......|..#.........|..#..#...#..#...|..#..#....#...|#...........#.",
		"#.....#.#.#.|...#.....#....|..#.#.#.#.#..|#...........|..#.........#.|..#.#.#.#...|..#.#.#.#...|#.......#.#.#.|..#.#.#.#.#...|...#....|......#.....|..#.#.#.......|..#.........|..#...#.#...#...|..#...#...#...|#...........#.",
		"#....#....#.|..#.#.#.#.#...|..#........#.|#...........|..#.........#.|..#.........|..#.........|#..........#..|..#.......#...|...#....|#.....#.....|..#....#......|..#.........|..#....#....#...|..#....#..#...|#...........#.",
		"#.....#.#.#.|..#.......#...|..#........#.|.#.......#..|..#........#..|..#.......#.|..#.........|.#.........#..|..#.......#...|...#....|#.....#.....|..#.....#.....|..#.......#.|..#.........#...|..#.....#.#...|.#.........#..",
		".#..........|#.#.#...#.#.#.|#.#.#.#.#.#..|..#.#.#.#...|#.#.#.#.#.#...|#.#.#.#.#.#.|#.#.........|..#.#.#.#.#...|#.#.#...#.#.#.|#.#.#.#.|.#.#.#......|#.#.....#.#.#.|#.#.#.#.#.#.|#.#.#.....#.#.#.|#.#.#.....#.#.|..#.#.#.#.#...",
		"..#.#.#.#...|..............|.............|............|..............|............|............|..............|..............|........|............|..............|............|................|..............|..............",
		"............|..............|.............|............|..............|............|............|..............|..............|........|............|..............|............|................|..............|.............."
	);

	ATWritePrinterPropFont(
		fontView, 0x50,
		"#.#.#.#.#.#.|..#.#.#.#...|#.#.#.#.#.#..|.#.#.#.#..|#.#.#.#.#.#.|#.#.#...#.#.#.|#.#.#...#.#.#.|#.#.#.....#.#.#.|.#.#.....#.#..|#.#.#...#.#.#.|#.#.#.#.|..#.#.#...|#.........|..#.#.#...|....#.....|..........",
		"..#........#|.#.......#..|..#........#.|#.......#.|#....#....#.|..#.......#...|..#.......#...|..#.........#...|....#...#.....|...#.....#....|#....#..|..#.......|..#.......|......#...|..#...#...|..........",
		"..#........#|#.........#.|..#........#.|#.........|.....#......|..#.......#...|..#.......#...|..#....#....#...|.....#.#......|....#...#.....|....#...|..#.......|...#......|......#...|#.......#.|..........",
		"..#.#.#.#.#.|#.........#.|..#.#.#.#.#..|.#.#.#.#..|.....#......|..#.......#...|...#.....#....|..#...#.#...#...|......#.......|.....#.#......|...#....|..#.......|....#.....|......#...|..........|..........",
		"..#.........|#.....#...#.|..#.....#....|........#.|.....#......|..#.......#...|....#...#.....|..#..#...#..#...|.....#.#......|......#.......|..#.....|..#.......|.....#....|......#...|..........|..........",
		"..#.........|.#......#...|..#......#...|#.......#.|.....#......|...#.....#....|.....#.#......|..#.#.....#.#...|....#...#.....|......#.......|.#....#.|..#.......|......#...|......#...|..........|..........",
		"#.#.#.......|..#.#.#...#.|#.#.......#.#|.#.#.#.#..|...#.#.#....|....#.#.#.....|......#.......|..#.........#...|.#.#.....#.#..|....#.#.#.....|#.#.#.#.|..#.#.#...|........#.|..#.#.#...|..........|#.#.#.#.#.",
		"............|............|.............|..........|............|..............|..............|................|..............|..............|........|..........|..........|..........|..........|..........",
		"............|............|.............|..........|............|..............|..............|................|..............|..............|........|..........|..........|..........|..........|.........."
	);

	ATWritePrinterPropFont(
		fontView, 0x60,
		".#....|..........|#.........|........|.......#..|..........|...#.#.#|..........|.#........|..#...|....#...|.#........|#.#...|..............|..........|..........",
		"..#...|..........|#.........|........|.......#..|..........|..#.....|..........|.#........|......|........|.#........|..#...|..............|..........|..........",
		"...#..|..#.#.#...|#.#.#.#...|.#.#.#.#|.#.#.#.#..|.#.#.#.#..|..#.....|..#.#.#.#.|.#.#.#.#..|#.#...|..#.#...|.#.....#..|..#...|.#.#.#.#.#.#..|.#.#.#.#..|.#.#.#.#..",
		"......|.......#..|#......#..|#.......|#......#..|#.......#.|#.#.#.#.|.#......#.|.#......#.|..#...|....#...|.#....#...|..#...|#.....#.....#.|.#......#.|#.......#.",
		"......|.#.#.#.#..|#......#..|#.......|#......#..|#.#.#.#.#.|..#.....|.#......#.|.#......#.|..#...|....#...|.#.#.#....|..#...|#.....#.....#.|.#......#.|#.......#.",
		"......|#......#..|#......#..|#.......|#......#..|#.........|..#.....|.#......#.|.#......#.|..#...|....#...|.#....#...|..#...|#.....#.....#.|.#......#.|#.......#.",
		"......|.#.#.#.#.#|#.#.#.#...|.#.#.#.#|.#.#.#.#..|.#.#.#.#..|..#.....|..#.#.#.#.|.#......#.|#.#.#.|....#...|.#.....#.#|#.#.#.|#.....#.....#.|.#......#.|.#.#.#.#..",
		"......|..........|..........|........|..........|..........|........|........#.|..........|......|....#...|..........|......|..............|..........|..........",
		"......|..........|..........|........|..........|..........|........|.#.#.#.#..|..........|......|.#.#....|..........|......|..............|..........|.........."
	);

	ATWritePrinterPropFont(
		fontView, 0x70,
		"..........|..........|........|..........|..#.....|..........|..........|..............|..........|..........|.........|....#.#.|..#..|#.#.....|..........",
		"..........|..........|........|..........|..#.....|..........|..........|..............|..........|..........|.........|..#.....|..#..|....#...|.#.#......",
		".#.#.#.#..|.#.#.#.#..|#...#.#.|.#.#.#.#..|#.#.#.#.|#.......#.|#.......#.|.#.........#..|#.#...#.#.|#.......#.|.#.#.#.#.|..#.....|..#..|....#...|#...#...#.",
		".#......#.|#......#..|.#.#....|#.........|..#.....|#.......#.|.#.....#..|.#....#....#..|...#.#....|.#.....#..|.....#...|#.......|..#..|......#.|.....#.#..",
		".#......#.|#......#..|.#......|.#.#.#.#..|..#.....|#.......#.|..#...#...|.#...#.#...#..|....#.....|..#...#...|....#....|..#.....|..#..|....#...|..........",
		".#.#.#.#..|.#.#.#.#..|.#......|........#.|..#.....|#.......#.|...#.#....|..#.#...#.#...|...#.#....|...#.#....|...#.....|..#.....|..#..|....#...|..........",
		".#........|.......#..|.#......|.#.#.#.#..|...#.#..|.#.#.#.#..|....#.....|...#.....#....|#.#...#.#.|....#.....|.#.#.#.#.|....#.#.|..#..|#.#.....|..........",
		".#........|.......#..|........|..........|........|..........|..........|..............|..........|...#......|.........|........|..#..|........|..........",
		".#........|.......#.#|........|..........|........|..........|..........|..............|..........|#.#.......|.........|........|.....|........|.........."
	);
}

extern constexpr ATPrinterFont825Prop g_ATPrinterFont825Prop = ATPrinterFont825Prop();

////////////////////////////////////////////////////////////////////////////////

consteval ATPrinterFont825Mono::ATPrinterFont825Mono()
	: ATPrinterFont{}
{
	ATPrinterFontView fontView(*this);

	ATWritePrinterFont(
		fontView, 0x20,
		".......|...#...|.#...#.|..#.#..|...#...|#.#...#|..#.#..|....#..|....#..|..#....|.......|.......|.......|.......|.......|......#",
		".......|...#...|.#...#.|..#.#..|.#.#.#.|#.#..#.|.#...#.|...#...|...#...|...#...|#..#..#|...#...|.......|.......|.......|.....#.",
		".......|...#...|.#...#.|#.#.#.#|#..#...|....#..|..#.#..|..#....|..#....|....#..|.#.#.#.|...#...|.......|.......|.......|....#..",
		".......|...#...|.......|..#.#..|.#.#.#.|...#...|.#.#...|.......|..#....|....#..|..#.#..|#.#.#.#|.......|#.#.#.#|.......|...#...",
		".......|.......|.......|#.#.#.#|...#..#|..#....|#...#..|.......|..#....|....#..|.#.#.#.|...#...|...#...|.......|...#...|..#....",
		".......|.......|.......|..#.#..|.#.#.#.|.#..#.#|#....#.|.......|...#...|...#...|#..#..#|...#...|..#.#..|.......|..#.#..|.#.....",
		".......|...#...|.......|..#.#..|...#...|#...#.#|.#.#..#|.......|....#..|..#....|.......|.......|...#...|.......|...#...|#......",
		".......|.......|.......|.......|.......|.......|.......|.......|.......|.......|.......|.......|..#....|.......|.......|......."
	);

	ATWritePrinterFont(
		fontView, 0x30,
		"..#.#..|...#...|.#.#.#.|.#.#.#.|...#.#.|#.#.#.#|....#..|#.#.#.#|.#.#.#.|.#.#.#.|...#...|...#...|......#|.......|#......|.#.#.#.",
		".#...#.|.#.#...|#.....#|#.....#|..#..#.|#......|...#...|.....#.|#.....#|#.....#|..#.#..|..#.#..|....#..|.......|..#....|#.....#",
		".#...#.|...#...|.....#.|......#|.#...#.|#.#.#..|..#....|....#..|#.....#|#.....#|...#...|...#...|..#....|#.#.#.#|....#..|......#",
		".#...#.|...#...|....#..|...#.#.|#.#.#.#|......#|.#.#.#.|...#...|.#.#.#.|.#.#.#.|.......|.......|#......|.......|......#|...#.#.",
		".#...#.|...#...|..#....|......#|.....#.|......#|#.....#|..#....|#.....#|....#..|...#...|...#...|..#....|#.#.#.#|....#..|...#...",
		".#...#.|...#...|.#.....|#.....#|.....#.|#.....#|#.....#|.#.....|#.....#|...#...|..#.#..|..#.#..|....#..|.......|..#....|.......",
		"..#.#..|.#.#.#.|#.#.#.#|.#.#.#.|.....#.|.#.#.#.|.#.#.#.|#......|.#.#.#.|..#....|...#...|...#...|......#|.......|#......|...#...",
		".......|.......|.......|.......|.......|.......|.......|.......|.......|.......|.......|..#....|.......|.......|.......|......."
	);

	ATWritePrinterFont(
		fontView, 0x40,
		".#.#.#.|...#...|#.#.#..|.#.#.#.|#.#.#..|#.#.#.#|#.#.#.#|.#.#.#.|#.....#|#.#.#.#|......#|#.....#|#......|#.....#|#.....#|.#.#.#.",
		"#.....#|..#.#..|#.....#|#.....#|#....#.|#......|#......|#.....#|#.....#|...#...|......#|#....#.|#......|.#...#.|.#....#|#.....#",
		"#...#.#|.#...#.|#.....#|#......|#.....#|#......|#......|#......|#.....#|...#...|......#|#...#..|#......|#.#.#.#|#.#...#|#.....#",
		"#..#..#|#.....#|#.#.#..|#......|#.....#|#.#.#..|#.#.#..|#......|#.#.#.#|...#...|......#|#.#....|#......|#..#..#|#..#..#|#.....#",
		"#..#.#.|#.#.#.#|#.....#|#......|#.....#|#......|#......|#...#.#|#.....#|...#...|#.....#|#...#..|#......|#.....#|#...#.#|#.....#",
		"#......|#.....#|#.....#|#.....#|#....#.|#......|#......|#.....#|#.....#|...#...|#.....#|#....#.|#......|#.....#|#....#.|#.....#",
		".#.#.#.|#.....#|#.#.#..|.#.#.#.|#.#.#..|#.#.#.#|#......|.#.#.#.|#.....#|#.#.#.#|.#.#.#.|#.....#|#.#.#.#|#.....#|#.....#|.#.#.#.",
		".......|.......|.......|.......|.......|.......|.......|.......|.......|.......|.......|.......|.......|.......|.......|......."
	);

	ATWritePrinterFont(
		fontView, 0x50,
		"#.#.#..|.#.#.#.|#.#.#..|.#.#.#.|#.#.#.#|#.....#|#.....#|#.....#|#.....#|#.....#|#.#.#.#|.#.#.#.|#......|.#.#.#.|...#...|.......",
		"#.....#|#.....#|#.....#|#.....#|...#...|#.....#|#.....#|#.....#|.#...#.|.#...#.|.....#.|.#.....|.#.....|.....#.|..#.#..|.......",
		"#.....#|#.....#|#.....#|#......|...#...|#.....#|#.....#|#.....#|..#.#..|..#.#..|....#..|.#.....|..#....|.....#.|.#...#.|.......",
		"#.#.#..|#.....#|#.#.#..|.#.#.#.|...#...|#.....#|#.....#|#..#..#|...#...|...#...|...#...|.#.....|...#...|.....#.|#.....#|.......",
		"#......|#...#.#|#...#..|......#|...#...|#.....#|.#...#.|#.#.#.#|..#.#..|...#...|..#....|.#.....|....#..|.....#.|.......|.......",
		"#......|#....#.|#....#.|#.....#|...#...|#.....#|..#.#..|.#...#.|.#...#.|...#...|.#.....|.#.....|.....#.|.....#.|.......|.......",
		"#......|.#.#..#|#.....#|.#.#.#.|...#...|.#.#.#.|...#...|#.....#|#.....#|...#...|#.#.#.#|.#.#.#.|......#|.#.#.#.|.......|#.#.#.#",
		".......|.......|.......|.......|.......|.......|.......|.......|.......|.......|.......|.......|.......|.......|.......|......."
	);

	ATWritePrinterFont(
		fontView, 0x60,
		"..#....|.......|#......|.......|......#|.......|.......|.......|#......|...#...|.......|#......|.#.#...|.......|.......|.......",
		"...#...|.......|#......|.......|......#|.......|...#.#.|.......|#......|.......|.....#.|#......|...#...|.......|.......|.......",
		"....#..|.#.#.#.|.#.#.#.|.#.#.#.|.#.#.#.|.#.#.#.|..#...#|.#.#.#.|.#.#.#.|.#.#...|.......|#...#..|...#...|.#...#.|#.#.#..|.#.#.#.",
		".......|......#|#.....#|#.....#|#.....#|#.....#|#.#.#..|#.....#|#.....#|...#...|.....#.|#..#...|...#...|#.#.#.#|#.....#|#.....#",
		".......|.#.#.#.|#.....#|#......|#.....#|#.#.#.#|..#....|#.....#|#.....#|...#...|.....#.|#.#....|...#...|#..#..#|#.....#|#.....#",
		".......|#.....#|#.....#|#......|#.....#|#......|..#....|.#.#.#.|#.....#|...#...|.....#.|#...#..|...#...|#.....#|#.....#|#.....#",
		".......|.#.#.#.|#.#.#..|.#.#.#.|..#.#.#|.#.#.#.|..#....|......#|#.....#|.#.#.#.|.#...#.|#.....#|.#.#.#.|#.....#|#.....#|.#.#.#.",
		".......|.......|.......|.......|.......|.......|.......|..#.#.#|.......|.......|..#.#..|.......|.......|.......|.......|......."
	);

	ATWritePrinterFont(
		fontView, 0x70,
		".......|.......|.......|.......|..#....|.......|.......|.......|.......|.......|.......|....#.#|...#...|#.#....|.......",
		".......|.......|.......|.......|..#....|.......|.......|.......|.......|.......|.......|..#....|...#...|....#..|..#....",
		"#.#.#..|.#.#.#.|#.#.#..|.#.#.#.|#.#.#..|#.....#|#.....#|#.....#|#.....#|#.....#|#.#.#.#|..#....|...#...|....#..|.#.#...",
		"#.....#|#....#.|.#...#.|#......|..#....|#.....#|#.....#|#.....#|.#...#.|#.....#|....#..|#......|...#...|......#|#...#.#",
		"#.....#|#....#.|.#.....|.#.#.#.|..#....|#.....#|.#...#.|#..#..#|...#...|.#.#.#.|...#...|..#....|...#...|....#..|.....#.",
		"#.#.#..|.#.#.#.|.#.....|......#|..#...#|#.....#|..#.#..|#.#.#.#|.#...#.|......#|..#....|..#....|...#...|....#..|.......",
		"#......|.....#.|.#.....|.#.#.#.|...#.#.|.#.#.#.|...#...|.#...#.|#.....#|......#|#.#.#.#|....#.#|...#...|#.#....|.......",
		"#......|......#|.......|.......|.......|.......|.......|.......|.......|.#.#.#.|.......|.......|...#...|.......|......."
	);
}

extern constexpr ATPrinterFont825Mono g_ATPrinterFont825Mono = ATPrinterFont825Mono();
